"""
Program: obsw_module_test.py
Date: 01.11.2019
Description: Module/High-Level test of on-board software, used by the TMTC client in
software test mode. It is very difficult to execute Unit Tests on embedded hardware.
The most significant functonalities for satellites are commanding and TM reception.
As such, these functionalities will be tested by sending TCs and checking the expected Telemetry.
Still, we make use of the Unit Testing Framework provided by Python (with some hacks involved).

Manual:
Set up the TMTC client as specified in the header comment and use the unit testing mode

For Developers:
TestService is the template method, analyseTmInfo and analyseTcInfo are implemented for
specific services or devices by setting up assertionDict entries which count received packets
based on subservice or entry values. There is a default implementation provided for analyseTcInfo.
The log files generated by individual G_SERVICE test or the software test mode when providing
the -p flag are very useful for extending the unit tester.
To check the dictionary keys of the telecommand and telemetry information queue entries,
go look up the packTmInformation() and packTcInformation() implementations in the TM and TC folder.

Example Service 17:
TC[17,1] and TC[17,128] are sent, TM[1,1] and TM[1,7] (TC verification) are expected twice,
TM[17,2] (ping reply) is expected twice and TM[5,1] (test event) is expected once.
Note that the TC verification counting is done by the template class by comparing with
TC source sequence count. For PUS standalone services (PUS Service Base),
only the start and completion success need to be asserted. For PUS gateway services
(PUS Commanding Service Base), step verification needs to be asserted additionally.
If there are commands with multiple steps, this needs
to be specified in the analyseTcInfo method of the child test.

@author: R. Mueller
"""
import unittest
from collections import deque
from typing import Deque

from tc.obsw_pus_tc_packer import packDummyDeviceTestInto
from tm.obsw_tm_service_1 import pusPacketInfoService1T
from abc import abstractmethod
from sendreceive.obsw_multiple_commands_sender_receiver import MultipleCommandSenderReceiver
from config import OBSW_Config as g
from tm.obsw_pus_tm_base import PusTmInfoQueueT, PusTmInfoT
from tc.obsw_pus_tc_base import PusTcInfoQueueT

TmInfoQueueService1T = Deque[pusPacketInfoService1T]


class TestService(unittest.TestCase):
    """
    Generic service test class.
    See documentation: https://docs.python.org/3/library/unittest.html#unittest.TestCase.setUpClass
    """
    @classmethod
    def setUpClass(cls):
        """
        A class method which is run before every individually class module is run
        must be decorated by the @classmethod fixture
        :return:
        """
        cls._displayMode = "long"
        # wait intervals between tc send bursts.
        # Example: [2,4] sends to send 2 tc from queue and wait, then sends another 2 and wait again
        cls.wait_intervals = []
        cls.wait_time = 5.0
        cls.print_tc = True
        # default wait time between tc send bursts

        cls.test_queue = deque()
        # Extremely ugly solution so that we can use the Python Unit Test Framework
        cls.tm_timeout = g.G_TM_TIMEOUT
        cls.tc_timeout_factor = g.G_TC_SEND_TIMEOUT_FACTOR
        cls.print_file = g.G_PRINT_TO_FILE
        cls.tmtc_printer = g.G_TMTC_PRINTER
        cls.tm_listener = g.G_TM_LISTENER
        cls.communication_interface = g.G_COM_INTERFACE
        # connectToBoard()

    def perform_testing_and_generate_assertion_dict(self):
        """
        This function should be called in each individual test to send the actual telecommands
        which are stored inside test_queue
        TODO: Maybe we should instantiate this once in the main and then reuse it instead
               of calling the constructor over and over again.
               If done so we need a setter for: wait_time, wait_intervals, printTm,
               tc_timeout_factor and the tc.queue. Furthermore, changing parameters should
               only be allowed as long as the commander/receiver is not running by checking a flag
        :return:
        """
        module_tester = MultipleCommandSenderReceiver(
            comInterface=self.communication_interface, tmtcPrinter=self.tmtc_printer,
            tmListener=self.tm_listener, tcQueue=self.test_queue, waitIntervals=self.wait_intervals,
            waitTime=self.wait_time, printTm=g.G_PRINT_RAW_TM)
        (tc_info_queue, tm_info_queue) = module_tester.send_tc_queue_and_return_info()
        assertion_dict = self._analyse_tm_tc_info(tm_info_queue, tc_info_queue)
        return assertion_dict

    def _analyse_tm_tc_info(
            self, tm_info_queue: PusTmInfoQueueT, tcInfoQueue: PusTcInfoQueueT) -> dict:
        self.tcSscArray = []
        self.tcServiceArray = []
        self.tcSubserviceArray = []

        # these number of TCs sent which need need to be verified
        # separate step counter if there is more than one step for a command !
        self.tcVerifyCounter = 0
        self.tcVerifyStepCounter = 0
        # child test implements this
        # default implementation provided
        self.analyseTcInfo(tcInfoQueue)

        # These values are incremented in the analyseTmInfo function
        # and compared to the counter values
        self.tcVerifiedStart = 0
        self.tcVerifiedCompletion = 0
        self.tcVerifiedStep = 0
        # The expected values are set in child test
        self.eventCounter = 0
        self.miscCounter = 0
        self.valid = True

        assertionDict = {}
        # child test implements this and can add additional dict entries
        self.analyseTmInfo(tm_info_queue, assertionDict)

        assertionDict.update({
            "TcStartCount": self.tcVerifiedStart,
            "TcCompletionCount": self.tcVerifiedCompletion,
            "TcStepCount": self.tcVerifiedStep,
            "EventCount": self.eventCounter,
            "MiscCount": self.miscCounter,
            "Valid": self.valid
        })

        return assertionDict

    def analyseTcInfo(self, tcInfoQueue: PusTcInfoQueueT):
        while not tcInfoQueue.__len__() == 0:
            currentTcInfo = tcInfoQueue.pop()
            self.tcVerifyCounter = self.tcVerifyCounter + 1
            # For commands with multiple steps, update this value manually !
            self.tcVerifyStepCounter = self.tcVerifyCounter + 1
            self.tcSscArray.append(currentTcInfo["ssc"])
            self.tcServiceArray.append(currentTcInfo["service"])
            self.tcSubserviceArray.append(currentTcInfo["subservice"])

    # must be implemented by child test !
    @abstractmethod
    def analyseTmInfo(self, tmInfoQueue: Deque, assertionDict: dict):
        pass

    # this function looks whether the tc verification SSC matched
    # a SSC of the sent TM
    def scanForRespectiveTc(self, currentTmInfo: PusTmInfoT):
        currentSubservice = currentTmInfo["subservice"]
        for possibleIndex, searchIndex in enumerate(self.tcSscArray):
            if searchIndex == currentTmInfo["tcSSC"]:
                if currentSubservice == 1:
                    self.tcVerifiedStart = self.tcVerifiedStart + 1
                elif currentSubservice == 5:
                    self.tcVerifiedStep = self.tcVerifiedStep + 1
                elif currentSubservice == 7:
                    self.tcVerifiedCompletion = self.tcVerifiedCompletion + 1

    def _perform_generic_assertion_test(self, assertionDict: dict):
        if assertionDict is None:
            print("Performing Generic Assertion Test: "
                  "Configuratrion error, asseretion dictionary was not passed properly")
            exit()
        self.assertEqual(assertionDict["TcStartCount"], self.tcVerifyCounter)
        self.assertEqual(assertionDict["TcCompletionCount"], self.tcVerifyCounter)
        self.assertEqual(assertionDict["EventCount"], self.eventExpected)
        self.assertEqual(assertionDict["MiscCount"], self.miscExpected)
        self.assertTrue(assertionDict["Valid"])

    # these tests are identical
    def performService5or17Test(self):
        assertionDict = self.perform_testing_and_generate_assertion_dict()
        self.eventExpected = 1
        self.miscExpected = 2
        self._perform_generic_assertion_test(assertionDict)

    def analyseService5or17TM(self, tmInfoQueue: PusTmInfoQueueT, assertionDict: dict):
        self.miscCounter = 0
        while not tmInfoQueue.__len__() == 0:
            currentTmInfo = tmInfoQueue.pop()
            # Tc verification scanning is generic and has been moved to the superclass
            if currentTmInfo["service"] == 1:
                self.scanForRespectiveTc(currentTmInfo)
            # Here, the desired event Id or RID can be specified
            if currentTmInfo["service"] == 5:
                if currentTmInfo["EventID"] == 8200 and currentTmInfo["RID"] == 0x51001700:
                    self.eventCounter = self.eventCounter + 1
            if currentTmInfo["service"] == 17:
                self.miscCounter = self.miscCounter + 1
            if currentTmInfo["valid"] == 0:
                self.valid = False
        assertionDict.update({"MiscCount": self.miscCounter})

    @classmethod
    def tearDownClass(cls):
        cls.eventCounter = 0
        cls.tcVerifyCounter = 0
        # noinspection PyUnresolvedReferences
        cls.test_queue.clear()


class TestDummyDevice(TestService):
    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        print("Testing Dummy Device")
        packDummyDeviceTestInto(super().test_queue)

    def analyseTmInfo(self, tmInfoQueue, assertionDict):
        pass


if __name__ == '__main__':
    unittest.main()
